/*
  --------------------------------------------------------------------------------------------------------------------
   Example sketch/program showing An Arduino Door Access Control featuring RFID, EEPROM, Relay
  ---------------------------------------------------------------------------------------------------------------------

  This example showing a complete Door Access Control System

  Simple Work Flow (not limited to) :
                                 +--------------------+
  +------------------------------>      READ TAGS     <-------------------------------------+
  |                              +--------------------+                                     |
  |                              |                    |                                     |
  |                         +----v-----+        +-----v----+                                |
  |                         |MASTER TAG|        |OTHER TAGS|                                |
  |                         +--+-------+        ++-------------+                            |
  |                            |                 |             |                            |
  |                      +-----v---+        +----v----+   +----v------+                     |
  |         +------------+READ TAGS+---+    |KNOWN TAG|   |UNKNOWN TAG|                     |
  |         |            +-+-------+   |    +-----------+ +------------------+              |
  |         |              |           |                |                    |              |
  |    +----v-----+   +----v----+   +--v--------+     +-v----------+  +------v----+         |
  |    |MASTER TAG|   |KNOWN TAG|   |UNKNOWN TAG|     |GRANT ACCESS|  |DENY ACCESS|         |
  |    +-----+----+   +---+-----+   +-----+-----+     +-----+------+  +-----+-----+         |
  |          |            |               |                 |               |               |
  |       +--v--+    +----v------+     +--v---+             |               +--------------->
  +-------+EXIT |    |DELETE FROM|     |ADD TO|             |                               |
          +-----+    |  DATABASE |     |  DB  |             |                               |
                     +-----------+     +------+             +-------------------------------+

   Use a Master Card which is act as Programmer then you can able to choose card holders who will granted access or not

 * **Easy User Interface**

   Just one RFID tag needed whether Delete or Add Tags. You can choose to use Leds for output or Serial LCD module to inform users.

 * **Stores Information on EEPROM**

   Information stored on non volatile Arduino's EEPROM memory to preserve Users' tag and Master Card. No Information lost
   if power lost. EEPROM has unlimited Read cycle but roughly 100,000 limited Write cycle.

    Typical pin layout used:
   -------------------------------------------------------------------------------------------------------
               MFRC522        esp32s3          Arduino       Arduino   Arduino    Arduino          Arduino
               Reader/PCD                 Uno/101       Mega      Nano v3    Leonardo/Micro   Pro Micro
   Signal      Pin              pin       Pin           Pin       Pin        Pin              Pin
   --------------------------------------------------------------------------------------------------------
   RST/Reset   RST              14         9             5         D9         RESET/ICSP-5     RST
   SPI SS      SDA(SS)          10        10            53        D10        10               10
   SPI MOSI    MOSI             11        11 / ICSP-4   51        D11        ICSP-4           16
   SPI MISO    MISO             12        12 / ICSP-1   50        D12        ICSP-1           14
   SPI SCK     SCK              13        13 / ICSP-3   52        D13        ICSP-3           15

*/

#define CONFIG_USB_CDC_ENABLED 1

#include <EEPROM.h>   // We are going to read and write cards IDs from/to EEPROM
#include <SPI.h>      // RC522 Module uses SPI protocol
#include <MFRC522.h>  // Library for Mifare RC522 Devices

#include <WiFi.h>
#include <HTTPClient.h>

#define EEPROM_SIZE 1024

#define SS_PIN 10
#define RST_PIN 14
MFRC522 mfrc522(SS_PIN, RST_PIN);  // Create MFRC522 instance.

#define Relay 41
#define Taster 1
#define Buzzer 42
#define wipeB 1  // Button pin for WipeMode
boolean wipeTaster, pwipeTaster;

boolean mstOnReader;
unsigned long mstDetTime;
boolean mstDetEE, mstNewMst;

#define NO_ACTION 0
#define ACCESS_GRANTED 1
#define ACCESS_DENIDED 2
#define PROGRAM_MODE 3
#define CARD_DELETED 4
#define CARD_ADDED 5
#define NEW_MASTER 6
#define NEW_MASTER_MODE 4000
#define DELETE_EEPROM 10000

#define SIG_DURATION 3000
#define SET_LED_TIME 250

#include "pitches.h"


//YWROBOT
//Compatible with the Arduino IDE 1.0
//Library version:1.1
#include <Wire.h> 
#include <LiquidCrystal_I2C.h>

LiquidCrystal_I2C lcd(0x27,16,2);  // set the LCD address to 0x27 for a 16 chars and 2 line display

String upperRow;
String lowerRow;
boolean refreshLCD = false;
long defLCD = 0;
#define REFRESH 5000


////////////////////Podaci i promjenljive za vezu sa bazom podataka
const char* ssid = "Tesla";
const char* password = "";

//Your Domain name with URL path or IP address with path
const char* serverName = "http://api.thingspeak.com/update"; //"http://192.168.1.106:1880/update";

int responseCode;
String responseContent;

String ReadTag;
int terminalID = 1;

String data;

//VAZNO!!!!!!
String server = "http://89.188.33.114";  //"192.168.1.209"; //"89.188.33.127"; //"192.168.100.8"; // www.example.com

String postData;
int pDataLen;

#define MAX_LENTH_OF_POST_REQUEST 512 /* Adjust the size to your needs */
String postRequest;
int pRqstLen;

// Keep this API Key value to be compatible with the PHP code provided in the project page.
// If you change the apiKeyValue value, the PHP file /post-esp-data.php also needs to have the same key
String apiKeyValue = "tPmAT5Ab3j7F9";

///////////////Kraj podataka i promjenljivih za vezu sa bazom podataka


// notes in the melody:
int AGmelody[] = {
  NOTE_DS5, NOTE_E5, NOTE_F5,
  NOTE_F5, NOTE_FS5, NOTE_G5, NOTE_G5, NOTE_GS5, NOTE_A5, NOTE_B5
};
// note durations: 4 = quarter note, 8 = eighth note, etc.:
int AGnoteDurations[] = {
  32, 32, 32,
  32, 32, 32, 32, 32, 16, 8
};

// notes in the melody:
int ADmelody[] = {
  NOTE_G4, NOTE_C4
};
// note durations: 4 = quarter note, 8 = eighth note, etc.:
int ADnoteDurations[] = {
  4, 4
};

unsigned char notes, currentNote;
int pauseBetweenNotes;
unsigned long lastNote;
int* melody;
int* noteDur;

boolean newCard;
int SigMod;
long detectionTime;
long setLED;

uint8_t successRead;  // Variable integer to keep if we have Successful Read from Reader

byte storedCard[8];  // Stores an ID read from EEPROM
byte readCard[8];    // Stores scanned ID read from RFID Module
byte masterCard[8];  //{0x3B, 0xD2, 0x2B, 0x1B}; // Master card's ID

boolean newMaster = false;

bool programMode = false;  // initialize programming mode

//Light signalisation
void Signals(int Mod) {
  static int noAct;

  if (Mod == NO_ACTION) {
    if (millis() - setLED > SET_LED_TIME) {
      if (noAct % 5) {
        neopixelWrite(RGB_BUILTIN, 0, 0, 0);  // All OFF
      } else if (noAct % 5 == 0) {
        neopixelWrite(RGB_BUILTIN, RGB_BRIGHTNESS, RGB_BRIGHTNESS, 0);  // Yellow
      }
      noAct++;
      setLED = millis();
      digitalWrite(Relay, LOW);
    }
  } else if (Mod == PROGRAM_MODE) {
    if (millis() - setLED > SET_LED_TIME) {
      if (!(noAct % 5)) {
        neopixelWrite(RGB_BUILTIN, RGB_BRIGHTNESS, 0, 0);  // Red
      } else if (noAct % 5) {
        neopixelWrite(RGB_BUILTIN, 0, RGB_BRIGHTNESS, 0);  // Green
      }
      noAct++;
      setLED = millis();
      digitalWrite(Relay, LOW);
    }
  } else if (Mod == ACCESS_GRANTED) {
    neopixelWrite(RGB_BUILTIN, 0, RGB_BRIGHTNESS, 0);  // Green
    digitalWrite(Relay, HIGH);
  } else if (Mod == ACCESS_DENIDED) {
    neopixelWrite(RGB_BUILTIN, RGB_BRIGHTNESS, 0, 0);  // Red
    digitalWrite(Relay, LOW);
  } else if (Mod == CARD_ADDED) {
    neopixelWrite(RGB_BUILTIN, 0, 2 * RGB_BRIGHTNESS, RGB_BRIGHTNESS);  // green+blue
    digitalWrite(Relay, LOW);
  } else if (Mod == CARD_DELETED) {
    neopixelWrite(RGB_BUILTIN, 2 * RGB_BRIGHTNESS, 0, RGB_BRIGHTNESS);  // Red+blue
    digitalWrite(Relay, LOW);
  } else if (Mod == NEW_MASTER) {
    neopixelWrite(RGB_BUILTIN, RGB_BRIGHTNESS, RGB_BRIGHTNESS, RGB_BRIGHTNESS);  // Red*green+Blue
    digitalWrite(Relay, LOW);
  }
}


//Compare to 4 bytes array
bool checkTwo(byte* a, byte* b) {
  for (uint8_t k = 0; k < 4; k++) {  // Loop 4 times
    if (a[k] != b[k]) {              // IF a != b then false, because: one fails, all fail
      return false;
    }
  }
  return true;
}


//Read card ID
uint8_t ReadCardID(byte* readTag) {
  if (!mfrc522.PICC_ReadCardSerial()) {  //Since a PICC placed get Serial and continue
    return 0;
  }
  // We support 4 byte card ID
  Serial.println(F("Scanned card ID:"));
  ReadTag = "";
  for (uint8_t i = 0; i < 4; i++) {  //
    readTag[i] = mfrc522.uid.uidByte[i];
    ReadTag.concat(String(readTag[i] < 0x10 ? "0" : ""));
    ReadTag.concat(String(readTag[i], HEX));
  }
  Serial.print(ReadTag);
  readTag[4] = '\0';
  Serial.println("");
  mfrc522.PICC_HaltA();  // Stop reading
  return 1;
}


// Check to see if the ID passed is the master programing card
bool isMaster(byte test[]) {
  return checkTwo(test, masterCard);
}


// Check master
int noMasterID(byte* mst) {
  for (uint8_t i = 0; i < 4; i++) {  // Loop 4 times to get the 4 Bytes
    if ((mst[i] != 0) && (mst[i] != 0xFF)) return (0);
  }
  return (1);
}


// Read an ID from EEPROM
void readID(uint8_t number, byte sCard[]) {
  uint8_t start = (number * 4) + 6;     // Figure out starting position
  for (uint8_t i = 0; i < 4; i++) {     // Loop 4 times to get the 4 Bytes
    sCard[i] = EEPROM.read(start + i);  // Assign values read from EEPROM to array
    delay(1);
  }
}

// Writing New Master ID to EEPROM
byte writeNewMasterID(byte* newMasterCard) {
  uint8_t start = 2;                            // Figure out where the next slot starts
  for (uint8_t j = 0; j < 4; j++) {             // Loop 4 times
    EEPROM.write(start + j, newMasterCard[j]);  // Write the array values to EEPROM in the right position
  }
  EEPROM.commit();
  Serial.println(F("Succesfully writed New Master ID record to EEPROM"));
  return (1);
}


// Reading Master ID from EEPROM
void readMasterID(byte* masterTag) {
  uint8_t start = 2;                        // Figure out starting position
  for (uint8_t i = 0; i < 4; i++) {         // Loop 4 times to get the 4 Bytes
    masterTag[i] = EEPROM.read(start + i);  // Assign values read from EEPROM to array
    Serial.print(masterTag[i], HEX);
    delay(1);
  }
  Serial.println();
  Serial.println(F("Succesfully read Master ID record from EEPROM"));
}

void writeToLCD(String *upRow, String *lwRow) {
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print(*upRow);
  lcd.setCursor(0, 1);
  lcd.print(*lwRow);
  Serial.println(*upRow);
  Serial.println(*lwRow);
}


///////////////////////// Funkcije za slnaje HTTP POST zahtjeva i analizu odgovora

// HTTP POST request sending
int httppost(String SerVer, String Resourse, String postData, String *Answer) {
  WiFiClient client;
  HTTPClient http;
    
  // Your Domain name with URL path or IP address with path
  http.begin(client, SerVer+Resourse);
      
  // If you need Node-RED/server authentication, insert user and password below
  //http.setAuthorization("REPLACE_WITH_SERVER_USERNAME", "REPLACE_WITH_SERVER_PASSWORD");
      
  // Specify content-type header
  http.addHeader(F("Content-Type"), F("application/x-www-form-urlencoded"));
  // Data to send with HTTP POST
  //String httpRequestData = String(F("api_key=XWH7ZA117WD2VX5D&field1=BME280&field2=24.25&field3=49.54&field4=1005.14"));           
  // Send HTTP POST request
  int httpResponseCode = http.POST(postData);
      
  // If you need an HTTP request with a content type: application/json, use the following:
  //http.addHeader("Content-Type", "application/json");
  //int httpResponseCode = http.POST("{\"api_key\":\"XWH7ZA117WD2VX5D\",\"sensor\":\"BME280\",\"value1\":\"24.25\",\"value2\":\"49.54\",\"value3\":\"1005.14\"}");

  // If you need an HTTP request with a content type: text/plain
  //http.addHeader("Content-Type", "text/plain");
  //int httpResponseCode = http.POST("Hello, World!");
     
  Serial.print(F("HTTP Response code: "));
  Serial.println(httpResponseCode);
  *Answer=http.getString();
  Serial.println(*Answer);  
        
  // Free resources
  http.end();

  return httpResponseCode;
}

// HTTP POST answer analizing
int httpAnswer(String Answer) {
  int fResponse=-1;

  Serial.println(Answer);
  if (Answer.indexOf(F("accepted")) != (-1)) {
    upperRow = String(F("Access granted!"));
    lowerRow = String(F("WELCOME"));
    refreshLCD = true;
    defLCD = millis();
    melody=AGmelody, noteDur=AGnoteDurations; notes=sizeof(AGmelody)/sizeof(int);  currentNote=0; pauseBetweenNotes=0; lastNote=0;
    currentNote = 0;
    fResponse = ACCESS_GRANTED;
  } 
  else if (Answer.indexOf(F("denided")) != (-1)) {
    upperRow = String(F("Access denided!"));
    lowerRow = String(F("SORRY!"));
    refreshLCD = true;
    defLCD = millis();
    fResponse = ACCESS_DENIDED;
    melody=ADmelody, noteDur=ADnoteDurations; notes=sizeof(ADmelody)/sizeof(int); currentNote=0; pauseBetweenNotes=0; lastNote=0;
    currentNote = 0;
  } 
  else if (Answer.indexOf(F("deleted")) != (-1)) {
    upperRow = String(F("Card"));
    lowerRow = String(F("deleted"));
    refreshLCD = true;
    defLCD = millis();
    melody=ADmelody, noteDur=ADnoteDurations; notes=sizeof(ADmelody)/sizeof(int); currentNote=0; pauseBetweenNotes=0; lastNote=0;
    currentNote = 0;
    fResponse = CARD_DELETED;
  } 
  else if (Answer.indexOf(F("added")) != (-1)) {
    upperRow = String(F("Card"));
    lowerRow = String(F("added"));
    refreshLCD = true;
    defLCD = millis();
    melody=AGmelody, noteDur=AGnoteDurations; notes=sizeof(AGmelody)/sizeof(int);  currentNote=0; pauseBetweenNotes=0; lastNote=0;
    currentNote = 0;
    fResponse = CARD_ADDED;
  } 
  else if (Answer.indexOf(F("Row created")) != (-1)) {
    upperRow = String(F("Row created"));
    lowerRow = String(F("succ..."));
    refreshLCD = true;
    defLCD = millis();
    melody=AGmelody, noteDur=AGnoteDurations; notes=sizeof(AGmelody)/sizeof(int);  currentNote=0; pauseBetweenNotes=0; lastNote=0;
    fResponse = CARD_ADDED;
  } 
  else if (Answer.indexOf(F("Error")) != (-1)) {
    upperRow = String(F("ERROR!!!"));
    lowerRow = String(F("ERROR!!!"));
    refreshLCD = true;
    defLCD = millis();
    melody=ADmelody, noteDur=ADnoteDurations; notes=sizeof(ADmelody)/sizeof(int); currentNote=0; pauseBetweenNotes=0; lastNote=0;
    fResponse = ACCESS_DENIDED;
  }
  return fResponse; 
}

///////////////////////// Kraj funkcija za rad sa esp modulom i komunikacijom sa bazom podataka


void setup() {
  //Konfigurisanje pinova
  pinMode(Relay, OUTPUT);
  pinMode(Taster, INPUT_PULLUP);
  pinMode(Buzzer, OUTPUT);

  Serial.begin(115200);  // Initiate a serial communication
  SPI.begin();           // Initiate  SPI bus
  mfrc522.PCD_Init();    // Initiate MFRC522

  // Allocate The Memory Size Needed
  EEPROM.begin(EEPROM_SIZE);

  // Read the Master tag(card) ID from EEPROM
  Serial.println();
  readMasterID(masterCard);
  Serial.println();

  //Inicijalizacija nacina signalizacije
  SigMod = NO_ACTION;
  newCard = true;

  // Inicijalizacija LCD-a
  lcd.begin(8, 9);  
  lcd.init();               
  // Print a message to the LCD.
  lcd.backlight();
  upperRow = String(F("LCD started!"));
  lowerRow = String(F("Wait ..."));  
  writeToLCD(&upperRow, &lowerRow);
    
  //Wipe taster chack - If the taster (wipeB) pressed longer than 10s it wipes EEPROM
  if (digitalRead(wipeB) == LOW || noMasterID(masterCard)) {  // when button pressed pin should get low, button connected to ground
    newMaster = true;
    if (newMaster) SigMod = NEW_MASTER;
    Serial.println(F("New MASTER mode started!!"));
    Serial.println(F("-----------------------------"));

    upperRow = String(F("Master mode"));
    lowerRow = String(F("Waiting card..."));
    writeToLCD(&upperRow, &lowerRow);
  }

  wipeTaster = digitalRead(wipeB);
  pwipeTaster = wipeTaster;

  mstOnReader = false;
  mstDetEE = false;
  mstNewMst = false;
  newCard = false;  /////////////

  //Povezivanje na WiFi pristupnu tačku
  WiFi.begin(ssid, password);
  Serial.println("Connecting");
  while(WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println("");
  Serial.print("Connected to WiFi network with IP Address: ");
  Serial.println(WiFi.localIP());

  //Pozdravna poruka nakon poverivanja na WiFi
  upperRow = String(F("Identify mode!"));
  lowerRow = String(F("WELCOME!"));
  writeToLCD(&upperRow, &lowerRow);
  refreshLCD = true;

  Serial.println(F("Približite identifikator čitaču..."));
  Serial.println();  
}

//Main loop
void loop() {
  // Look for new cards
  if (mfrc522.PICC_IsNewCardPresent()) {
    detectionTime = millis();
    successRead = ReadCardID(readCard);
    if (!successRead) return;
    if (!mstOnReader) {
      if (newMaster) {
        Serial.println(F("Writing new Master ID"));
        Serial.println(F("-----------------------------"));
        writeNewMasterID(readCard);
        //Read the Master tag(card) ID from EEPROM
        readMasterID(masterCard);
        newMaster = false;
        melody=AGmelody, noteDur=AGnoteDurations; notes=sizeof(AGmelody)/sizeof(int);  currentNote=0; pauseBetweenNotes=0; lastNote=0;
        Serial.println(F("Hello Master - Entered Identification Mode"));        
      } 
      else if (programMode) {
        SigMod = PROGRAM_MODE;
        if (isMaster(readCard)) {  //When in program mode check First If master card scanned again to exit program mode
          Serial.println(F("Master Card Scanned"));
          Serial.println(F("Exiting Program Mode"));
          Serial.println(F("-----------------------------"));

          upperRow = String(F("Master again"));
          lowerRow = String(F("Identify mode!"));
          refreshLCD = true;
          defLCD = millis();

          programMode = false;
          SigMod = NO_ACTION;
          mstOnReader = true; mstDetTime = millis(); mstDetEE = true; mstNewMst = true;          
        } 
        else {
          // data sent must be under this form //name1=value1&name2=value2.
          postData = String(F("api_key=")) + apiKeyValue + F("&terminalID=") + terminalID;
          postData += String(F("&tagID=")) + ReadTag + "";

          upperRow = String(F("Card Scanned"));
          lowerRow = String(F("Ins/del user..."));
          writeToLCD(&upperRow, &lowerRow);
          defLCD = millis();

          responseCode=httppost(server, String(F("/idsysev/post_exercise.php")), postData, &responseContent);
          //Predhodnu liniji koda, za vježbu zamijeniti s linijom koda koja je u nastavku komentara, u liniji 539.
          //responseCode=httppost(server, String(F("/idsysev/post_insdel_user.php")), postData, &responseContent);
          if(responseCode==200){
            SigMod=httpAnswer(responseContent);
          }
          else {
            Serial.println(responseCode);
            Serial.println(responseContent);

            upperRow = String("Error")+": "+String(responseCode);
            lowerRow = String("Wrong respond");
            writeToLCD(&upperRow, &lowerRow);
            defLCD = millis();                

            melody=ADmelody, noteDur=ADnoteDurations; notes=sizeof(ADmelody)/sizeof(int); currentNote=0; pauseBetweenNotes=0; lastNote=0;
            currentNote = 0;            
          }
          detectionTime = millis();
        }
      } 
      else {
        if (isMaster(readCard)) {  // If scanned card's ID matches Master Card's ID - enter program mode
          programMode = true;
          Serial.println(F("Hello Master - Entered Program Mode"));
          uint8_t count = EEPROM.read(0);  // Read the first Byte of EEPROM that
          Serial.print(F("I have "));      // stores the number of ID's in EEPROM
          Serial.print(count);
          Serial.print(F(" record(s) on EEPROM"));
          Serial.println("");
          Serial.println(F("Scan a tag(card) to ADD or REMOVE to EEPROM"));
          Serial.println(F("Scan Master tag(card) again to Exit Program Mode"));
          Serial.println(F("-----------------------------"));

          upperRow = String(F("Master Scanned"));
          lowerRow = String(F("Program mode!"));
          refreshLCD = true;
          defLCD = millis();

          SigMod = PROGRAM_MODE;
          mstOnReader = true; mstDetTime = millis(); mstDetEE = true; mstNewMst = true;          
        } 
        else {
          // data sent must be under this form //name1=value1&name2=value2.
          postData = String(F("api_key=")) + apiKeyValue + F("&terminalID=") + terminalID;
          postData += String(F("&tagID=")) + ReadTag + "";

          upperRow = String(F("Card Scanned"));
          lowerRow = String(F("Check in prog..."));
          writeToLCD(&upperRow, &lowerRow);
          defLCD = millis();
                  
          responseCode=httppost(server, String(F("/idsysev/post_exercise.php")), postData, &responseContent);
          //Predhodnu liniji koda, za vježbu zamijeniti s linijom koda koja je u nastavku komentara, u liniji 582
          //responseCode=httppost(server, String(F("/idsysev/post_chkusr_evid.php")), postData, &responseContent);
          if(responseCode==200){
            SigMod=httpAnswer(responseContent);
          }
          else {
            Serial.println(responseCode);
            Serial.println(responseContent);

            upperRow = String("Error")+": "+String(responseCode);
            lowerRow = String("Wrong respond");
            writeToLCD(&upperRow, &lowerRow);
            defLCD = millis();

            melody=ADmelody, noteDur=ADnoteDurations; notes=sizeof(ADmelody)/sizeof(int); currentNote=0; pauseBetweenNotes=0; lastNote=0;
            currentNote = 0;               
          }
          detectionTime = millis();
        }
      }
    }
    if (!mstOnReader) {
      // Halt PICC
      mfrc522.PICC_HaltA();     
    } 
    else {
      mfrc522.PCD_Init();     
    }
  }
  else if (mstOnReader) {
    mstOnReader = false;
    mstDetEE=false; mstNewMst=false;
    //Serial.println(F("There is no Master card!"));    
  }

  //Wipe taster chack - If the taster (wipeB) pressed longer than 10s it wipes EEPROM
   if (mstNewMst && (millis()- mstDetTime >= NEW_MASTER_MODE)) {
    mstNewMst=false;
    newMaster = true;
    Serial.println(F("New MASTER mode started!!"));
    Serial.println(F("-----------------------------"));

    upperRow = String(F("Master mode"));
    lowerRow = String(F("Waiting card..."));
    refreshLCD = true;
    defLCD = millis();

    SigMod = NEW_MASTER;
    melody=AGmelody, noteDur=AGnoteDurations; notes=sizeof(AGmelody)/sizeof(int);  currentNote=0; pauseBetweenNotes=0; lastNote=0;
    programMode = false;
  }

  //Wipe taster chack - If the taster (wipeB) pressed longer than 10s it wipes EEPROM
  wipeTaster = digitalRead(wipeB);
  if ((wipeTaster == LOW) && (wipeTaster != pwipeTaster)) {
    newMaster = !newMaster;
    if (newMaster) {
      Serial.println(F("New MASTER mode started!!"));
      Serial.println(F("-----------------------------"));

      upperRow = String(F("Master mode"));
      lowerRow = String(F("Waiting card..."));
      refreshLCD = true;
      defLCD = millis();

      SigMod = NEW_MASTER;
      melody=AGmelody, noteDur=AGnoteDurations; notes=sizeof(AGmelody)/sizeof(int);  currentNote=0; pauseBetweenNotes=0; lastNote=0;
      programMode = false;      
    }
  }
  pwipeTaster = wipeTaster;

  //Signalisation
  if (millis() - detectionTime > SIG_DURATION) {
    if (newMaster) SigMod = NEW_MASTER;
    else if (programMode) SigMod = PROGRAM_MODE;
    else SigMod = NO_ACTION;
  } 
  Signals(SigMod);

  //Provjera treba li vratiti default ispis
  if (((millis() - defLCD) > REFRESH) && defLCD) {
    if (programMode) upperRow = String(F("Program mode!"));
    else upperRow = String(F("Identify mode!"));
    lowerRow = String(F("WELCOME!"));
    defLCD = 0;
    refreshLCD = true;
  }

  //Ispis na LCD-u
  if (refreshLCD) {
    writeToLCD(&upperRow, &lowerRow);
    refreshLCD = false;
  }

  //Zvučni signal
  if (currentNote < notes)
  {
    if(millis()-lastNote > pauseBetweenNotes){
      lastNote=millis();
      int duration = 1000 / noteDur[currentNote];
      tone(Buzzer, melody[currentNote],duration);    
      currentNote++;
      pauseBetweenNotes = duration * 1.30;
    }
  }
}
